﻿using System.Collections;
using System.Collections.Generic;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

//CarSelector is the UI used in the in-game garages
public class CarSelector : MonoBehaviour
{
    public GameObject ParentSelectorUI;
    public float RotationSpeed = 20.0f;

    //UI elements
    public TMP_Text MoneyText;
    public TMP_Text CarNameText;
    public TMP_Text PriceText;
    public GameObject CarObject;
    public Image CarImage;
    public ProgressBar SpeedBar;
    public ProgressBar AccelerationBar;
    public ProgressBar HandlingBar;
    public CustomButton BuyButton;
    public CustomButton CancelButton;
    public Animator Animator;

    //Current state variables
    private int _CurrentIndex = 0;
    private float _CurrentRotation;
    private List<GameObject> _Cars = new List<GameObject>();
    private bool _ShowingMessageBox = false;

    /// <summary>
    /// Shows the car selector UI with all the cars for the specific class type
    /// </summary>
    /// <param name="carType"></param>
    public void Show(Constants.CarType carType)
    {
        if (!gameObject.activeSelf)
        {
            //If we're not active already, set up the UI based on the class type
            gameObject.SetActive(true);
            MoneyText.SetText("$" + GameManager.Instance.Money.ToString());

            switch (carType)
            {
                case Constants.CarType.LowEnd:
                    SetCars(CarsManager.Instance.LowEndCars);
                    break;

                case Constants.CarType.MediumEnd:
                    SetCars(CarsManager.Instance.MediumEndCars);
                    break;

                case Constants.CarType.HighEnd:
                    SetCars(CarsManager.Instance.HighEndCars);
                    break;
            }

            //Make sure we handle the audio correctly, stop the current track and play our BGM
            AudioManager.Instance.StopFile(GameManager.Instance.CurrentMusicFile);
            AudioFile carSelectionFile = AudioManager.Instance.LoadedFiles["CarSelectionBGM"];
            GameManager.Instance.CurrentMusicFile = carSelectionFile;
            AudioManager.Instance.PlayFile(GameManager.Instance.CurrentMusicFile);
            AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIPageTransToAudioID]);
            
            //Now update and pause the main game, and play the animation
            GameManager.Instance.SetGameState(Constants.GameState.ShowingCarSelection);
            Time.timeScale = 0.0f;
            Animator.Play("Show");
        }
    }

    /// <summary>
    /// Hides the UI
    /// </summary>
    public void Hide()
    {
        AudioManager.Instance.PlayFile(AudioManager.Instance.LoadedFiles[Constants.UIPageTransFromAudioID]);
        Animator.Play("Hide");
    }

    /// <summary>
    /// Handler for the left arrow being clicked, updates the current car to view and plays the UI SFX
    /// </summary>
    public void OnLeftArrowClicked()
    {
        if (_CurrentIndex - 1 < 0)
        {
            _CurrentIndex = _Cars.Count;
        }

        _CurrentIndex--;
        ShowCar(_CurrentIndex);
        AudioManager.Instance.PlaySFX(AudioManager.Instance.LoadedFiles[Constants.UIButtonFocusAudioID].Clip);
    }

    /// <summary>
    /// Handler for the right arrow being clicked, updates the current car to view and plays the UI SFX
    /// </summary>
    public void OnRightArrowClicked()
    {
        if (_CurrentIndex + 1 == _Cars.Count)
        {
            _CurrentIndex = -1;
        }

        _CurrentIndex++;
        ShowCar(_CurrentIndex);
        AudioManager.Instance.PlaySFX(AudioManager.Instance.LoadedFiles[Constants.UIButtonFocusAudioID].Clip);
    }

    /// <summary>
    /// Handler for the buy button clicked, checks the price for safety, then updates the current car and hides the UI
    /// </summary>
    public async void OnBuyButtonClicked()
    {
        CarDefinition carDefinition = _Cars[_CurrentIndex].GetComponent<CarDefinition>();

        if (carDefinition.Price > GameManager.Instance.Money)
        {
            return; //We can't afford it, let's stop and return
        }

        else
        {
            //Let's make the user explicitly confirm with the message box
            gameObject.Disable();
            _ShowingMessageBox = true;
            int button = await MessageBoxManager.Instance.Show("ARE YOU SURE?", string.Format("ARE YOU SURE YOU WANT TO BUY {0} FOR {1}?", carDefinition.Name, carDefinition.Price), new List<string>() { "NO, CANCEL", "YES, BUY IT" });
            _ShowingMessageBox = false;
            gameObject.Enable();

            if (button == 1)
            {
                //They pressed yes, so let's decrement their money based on the cost, set the new car as current and hide the UI
                GameManager.Instance.IncrementMoney(-carDefinition.Price);
                GameManager.Instance.SetPlayerCar(_Cars[_CurrentIndex]);
                Hide();
            }
        }
    }

    /// <summary>
    /// Handler for the cancel button clicked, simply hides the UI
    /// </summary>
    public void OnCancelButtonClicked()
    {
        Hide();
    }

    /// <summary>
    /// Handler for the hide animation completed, restores the running state of the game
    /// </summary>
    public void OnHideAnimationCompleted()
    {
        gameObject.SetActive(false);
        GameManager.Instance.SetGameState(Constants.GameState.Playing);
        Time.timeScale = 1.0f;
        GameManager.Instance.RestartAmbientMusic();
    }

    /// <summary>
    /// Sets up the internal list of cars, resets the state and shows it
    /// </summary>
    /// <param name="cars">The list of cars available to show</param>
    private void SetCars(List<GameObject> cars)
    {
        _Cars = cars;
        _CurrentIndex = 0;
        ShowCar(_CurrentIndex);
    }

    /// <summary>
    /// Shows the car at the set index in the list in the UI
    /// </summary>
    /// <param name="index">The index of the car in the list of cars</param>
    private void ShowCar(int index)
    {
        if (index < _Cars.Count)
        {
            //Set the rotation here so we don't "jankily" snap back to 0 when we change
            CarObject.transform.eulerAngles = new Vector3(0.0f, 0.0f, _CurrentRotation);

            //Set all the UI up based on the car's properties
            CarDefinition car = _Cars[index].GetComponent<CarDefinition>();
            CarNameText.text = "<color=#FF7F00>" + car.NamePrefix + "</color> " + car.NameSuffix;
            PriceText.text = "$" + car.Price.ToString();
            CarImage.sprite = car.GetComponent<SpriteRenderer>().sprite;
            CarImage.color = car.GetComponent<CarDefinition>().Colour;
            SpeedBar.SetProgressPercentage(car.GetSpeedPercentage());
            AccelerationBar.SetProgressPercentage(car.GetAccelerationPercentage());
            HandlingBar.SetProgressPercentage(car.GetHandlingPercentage());

            //Set the state of the button based on the money available
            if(car.Price > GameManager.Instance.Money)
            {
                BuyButton.Disable();
            }

            else
            {
                BuyButton.Enable();
            }
        }
    }

    private void Update()
    {
        //Update the money and rotation for the animation
        MoneyText.SetText("$" + GameManager.Instance.Money.ToString());
        _CurrentRotation -= RotationSpeed * Time.unscaledDeltaTime;
        CarObject.transform.eulerAngles = new Vector3(0.0f, 0.0f, _CurrentRotation);

        //Detect whether we should hide the UI
        if (Input.GetKeyDown(KeyCode.Escape) && !_ShowingMessageBox)
        {
            Hide();
        }
    }
}
